<?php
/**
 * Handles all public-facing (front-end) features of the plugin.
 */
class WIC_Public {

    private $version;

    public function __construct( $version ) {
        $this->version = $version;
        $this->load_hooks();
    }

    private function load_hooks() {
        add_action( 'woocommerce_single_product_summary', [ $this, 'display_installment_info_hook' ], 25 );
        add_shortcode( 'wic_details', [ $this, 'render_installment_shortcode' ] );
        add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_styles_and_scripts' ] );
        add_action( 'wp_footer', [ $this, 'render_modal_html' ] );
    }

    public function enqueue_styles_and_scripts() {
        if ( is_product() && get_post_meta( get_the_ID(), '_wic_enabled', true ) === 'yes' ) {
            wp_enqueue_style( 'wic-public-style', WIC_URL . 'assets/css/wic-public-style.css', [], $this->version );
            
            $product_id = get_the_ID();
            $highlight_color = get_post_meta( $product_id, '_wic_highlight_color', true ) ?: get_option('wic_default_highlight_color', '#28a745');
            $contact_method = get_post_meta( $product_id, '_wic_contact_method', true ) ?: get_option('wic_default_contact_method', 'none');
            
            $button_color = $highlight_color; // Default to highlight color
            if (strpos($contact_method, 'custom_') === 0) {
                $index = (int) str_replace('custom_', '', $contact_method);
                $messengers = get_option('wic_custom_messengers', []);
                if (isset($messengers[$index]['color']) && !empty($messengers[$index]['color'])) {
                    $button_color = $messengers[$index]['color'];
                }
            } elseif ($contact_method === 'telegram') {
                $button_color = '#0088cc';
            }

            $custom_css = ":root { --wic-success-color: " . esc_attr($highlight_color) . "; --wic-contact-button-color: " . esc_attr($button_color) . "; }";
            wp_add_inline_style( 'wic-public-style', $custom_css );

            wp_enqueue_script( 'wic-public-script', WIC_URL . 'assets/js/wic-public-script.js', [], $this->version, true );
        }
    }

    public function display_installment_info_hook() {
        $display_method = get_post_meta( get_the_ID(), '_wic_display_method', true ) ?: get_option('wic_default_display_method', 'auto');
        if ($display_method !== 'shortcode') {
            echo $this->get_installment_info_html();
        }
    }
    
    public function render_installment_shortcode() {
        return $this->get_installment_info_html();
    }

    private function get_installment_info_html() {
        if ( !is_product() || get_post_meta( get_the_ID(), '_wic_enabled', true ) !== 'yes' ) return '';
        $product = wc_get_product( get_the_ID() );
        if (!$product || $product->get_price() <= 0) return '';
        
        $settings = $this->get_product_settings(get_the_ID());
        if ($settings['installments'] <= 0) return '';

        $calc = $this->calculate_installments($product->get_price(), $settings);

        // Determine if the contact button should be displayed
        $has_contact_button = ($settings['contact_method'] !== 'none' && !empty($settings['contact_url']));
        
        $wrapper_class = 'wic-buttons-wrapper';
        if (!$has_contact_button) {
            $wrapper_class .= ' wic-single-button-wrapper';
        }

        ob_start();
        ?>
        <div class="wic-info-container">
            <div class="wic-summary">
                <div class="wic-summary-item">
                    <span class="label"><?php _e( 'پیش‌پرداخت:', WIC_TEXT_DOMAIN ); ?></span>
                    <span class="value"><?php echo wc_price( $calc['down_payment'] ); ?></span>
                </div>
                <div class="wic-summary-item">
                    <span class="label"><?php echo sprintf( __( '%d قسط ماهانه:', WIC_TEXT_DOMAIN ), $settings['installments'] ); ?></span>
                    <span class="value primary"><?php echo wc_price( $calc['monthly_payment'] ); ?></span>
                </div>
            </div>
            <div class="<?php echo esc_attr($wrapper_class); ?>">
                <button id="wic-calc-open-btn" class="button alt wic-calc-button"><?php _e( 'مشاهده جزئیات', WIC_TEXT_DOMAIN ); ?></button>
                <?php if ($has_contact_button): ?>
                    <a href="<?php echo esc_url($settings['contact_url']); ?>" target="_blank" class="button alt wic-calc-button <?php echo esc_attr($settings['button_class']); ?>">
                        <?php _e('ثبت خرید اقساطی', WIC_TEXT_DOMAIN); ?>
                    </a>
                <?php endif; ?>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    public function render_modal_html() {
        if ( !is_product() || get_post_meta( get_the_ID(), '_wic_enabled', true ) !== 'yes' ) return;
        $product = wc_get_product( get_the_ID() );
        if (!$product || $product->get_price() <= 0) return;

        $settings = $this->get_product_settings(get_the_ID());
        $calc = $this->calculate_installments($product->get_price(), $settings);
        ?>
        <div id="wic-calculator-modal" class="wic-modal">
            <div class="wic-modal-content">
                <div class="wic-modal-header">
                    <h3><?php _e( 'جزئیات فروش اقساطی', WIC_TEXT_DOMAIN ); ?></h3>
                    <span class="wic-modal-close">&times;</span>
                </div>
                <div class="wic-modal-body">
                    <div class="wic-results-wrapper">
                        <div class="wic-result-item"><span class="label"><?php _e( 'قیمت نقدی کالا:', WIC_TEXT_DOMAIN ); ?></span><span class="value"><?php echo wc_price( $product->get_price() ); ?></span></div>
                        <div class="wic-result-item"><span class="label"><?php _e( 'مبلغ پیش‌پرداخت:', WIC_TEXT_DOMAIN ); ?></span><span class="value"><?php echo wc_price( $calc['down_payment'] ); ?></span></div>
                        <div class="wic-result-item"><span class="label"><?php _e( 'مبلغ وام (باقیمانده):', WIC_TEXT_DOMAIN ); ?></span><span class="value"><?php echo wc_price( $calc['principal'] ); ?></span></div>
                        <div class="wic-result-item"><span class="label"><?php _e( 'تعداد اقساط:', WIC_TEXT_DOMAIN ); ?></span><span class="value"><?php echo $settings['installments'] . ' ' . __( 'ماه', WIC_TEXT_DOMAIN ); ?></span></div>
                        <div class="wic-result-item"><span class="label"><?php _e( 'مبلغ هر قسط:', WIC_TEXT_DOMAIN ); ?></span><span class="value primary"><?php echo wc_price( $calc['monthly_payment'] ); ?></span></div>
                        <div class="wic-result-item"><span class="label"><?php _e( 'مجموع سود:', WIC_TEXT_DOMAIN ); ?></span><span class="value interest"><?php echo wc_price( $calc['total_interest'] ); ?></span></div>
                        <div class="wic-result-item total"><span class="label"><?php _e( 'مبلغ نهایی پرداختی شما:', WIC_TEXT_DOMAIN ); ?></span><span class="value"><?php echo wc_price( $calc['total_payment'] ); ?></span></div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    private function get_product_settings($product_id) {
        $settings = [
            'down_payment_percent' => (float) (get_post_meta($product_id, '_wic_down_payment_percent', true) ?: get_option('wic_default_down_payment', 20)),
            'installments' => (int) (get_post_meta($product_id, '_wic_installments', true) ?: get_option('wic_default_installments', 12)),
            'annual_rate' => (float) (get_post_meta($product_id, '_wic_interest_rate', true) ?: get_option('wic_default_interest_rate', 23)),
            'contact_method' => get_post_meta($product_id, '_wic_contact_method', true) ?: get_option('wic_default_contact_method', 'none'),
            'contact_url' => '',
            'button_class' => ''
        ];

        $product_name = get_the_title($product_id);
        $message = rawurlencode(sprintf('سلام، من مایلم محصول «%s» را به صورت اقساطی خریداری کنم.', $product_name));

        if ($settings['contact_method'] === 'whatsapp') {
            $number = get_post_meta($product_id, '_wic_whatsapp_number', true) ?: get_option('wic_default_whatsapp_number');
            if ($number) { $settings['contact_url'] = 'https://wa.me/' . $number . '?text=' . $message; $settings['button_class'] = 'whatsapp'; }
        } elseif ($settings['contact_method'] === 'telegram') {
            $id = get_post_meta($product_id, '_wic_telegram_id', true) ?: get_option('wic_default_telegram_id');
            if ($id) { $settings['contact_url'] = 'https://t.me/' . $id; $settings['button_class'] = 'telegram'; }
        } elseif (strpos($settings['contact_method'], 'custom_') === 0) {
            $index = (int) str_replace('custom_', '', $settings['contact_method']);
            $messengers = get_option('wic_custom_messengers', []);
            if (isset($messengers[$index])) {
                $messenger = $messengers[$index];
                $custom_ids = get_post_meta($product_id, '_wic_custom_ids', true) ?: [];
                $contact_info = $custom_ids[$index] ?? '';
                if (!empty($contact_info)) {
                    $url = str_replace('{ID}', $contact_info, $messenger['link']);
                    if (!empty($messenger['text_support'])) { $url .= (strpos($url, '?') === false ? '?' : '&') . 'text=' . $message; }
                    $settings['contact_url'] = $url;
                    $settings['button_class'] = 'custom-messenger';
                }
            }
        }
        return $settings;
    }

    private function calculate_installments($price, $settings) {
        $down_payment = round($price * ($settings['down_payment_percent'] / 100), wc_get_price_decimals());
        $principal = $price - $down_payment;
        $monthly_rate = $settings['annual_rate'] / 12 / 100;
        
        $monthly_payment = 0;
        if ($settings['installments'] > 0) {
            if ($monthly_rate === 0.0) {
                $monthly_payment = $principal / $settings['installments'];
            } else {
                $factor = pow(1 + $monthly_rate, $settings['installments']);
                $monthly_payment = ($factor != 1) ? ($principal * ($monthly_rate * $factor) / ($factor - 1)) : 0;
            }
        }

        $total_interest = ($monthly_payment * $settings['installments']) - $principal;
        $total_payment = $price + $total_interest;

        return ['down_payment' => $down_payment, 'principal' => $principal, 'monthly_payment' => $monthly_payment, 'total_interest' => $total_interest, 'total_payment' => $total_payment];
    }
}
