(function($) {
    $(document).ready(function() {

        // --- Create and append modal to body ONCE ---
        if ($('#apc-modal-overlay').length === 0) {
            const modalHTML = `
                <div id="apc-modal-overlay" class="apc-modal-overlay">
                    <div class="apc-modal-content">
                        <p id="apc-modal-text"></p>
                        <button id="apc-modal-close-btn" class="apc-modal-close-btn">متوجه شدم</button>
                    </div>
                </div>`;
            $('body').append(modalHTML);

            // Setup modal listeners ONCE
            $('#apc-modal-close-btn').on('click', function() {
                $('#apc-modal-overlay').css('display', 'none');
            });
            $('#apc-modal-overlay').on('click', function(e) {
                if ($(e.target).is('#apc-modal-overlay')) {
                    $(this).css('display', 'none');
                }
            });
        }
        
        function showModal(message) {
            $('#apc-modal-text').text(message);
            $('#apc-modal-overlay').css('display', 'flex');
        }

        // --- Process EACH calculator instance separately ---
        $('.apc-container').each(function() {
            const container = $(this);

            // --- Elements within this specific container ---
            const formSection = container.find('.apc-form-section');
            const loadingSection = container.find('.apc-loading-section');
            const resultSection = container.find('.apc-result-section');
            const calculateBtn = container.find('.apc-calculate-btn');
            const amazonUrlInput = container.find('.apc-amazon-url-input');
            const aedPriceInput = container.find('.apc-aed-price-input');
            const whatsappBtn = container.find('.apc-whatsapp-btn');
            const airShippingOption = container.find('.apc-air-shipping-option');
            const seaShippingOption = container.find('.apc-sea-shipping-option');

            // --- Settings from WordPress (available globally) ---
            const settings = apc_settings;

            let currentProductData = {};
            let selectedShippingMethod = 'air';

            function updateShippingSelectionVisuals() {
                airShippingOption.addClass('apc-selected');
                seaShippingOption.removeClass('apc-selected');
                if (selectedShippingMethod === 'sea') {
                    seaShippingOption.addClass('apc-selected');
                    airShippingOption.removeClass('apc-selected');
                }
            }

            function displayResults(productData, finalPrice) {
                resultSection.find('.apc-product-price-aed').text(`${productData.priceAED.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })} ${settings.currency_symbol}`);
                
                const exchangeRate = parseFloat(settings.exchange_rate);
                resultSection.find('.apc-exchange-rate-display').text(`(نرخ هر ${settings.currency_symbol}: ${exchangeRate.toLocaleString('fa-IR')} تومان)`);
                
                const finalPriceFormatted = `${Math.round(finalPrice).toLocaleString('fa-IR')} تومان`;
                resultSection.find('.apc-air-price').text(finalPriceFormatted);
                resultSection.find('.apc-sea-price').text(finalPriceFormatted);
                
                currentProductData.title = productData.title;
                currentProductData.priceAED = productData.priceAED;
                currentProductData.finalPriceToman = Math.round(finalPrice).toLocaleString('fa-IR');

                selectedShippingMethod = 'air';
                updateShippingSelectionVisuals();

                loadingSection.hide();
                resultSection.show();
            }

            // --- Event Listeners for THIS instance ---
            calculateBtn.on('click', function() {
                const url = amazonUrlInput.val();
                const priceString = aedPriceInput.val();
                
                if (!url || !url.startsWith('http') || !url.includes('amazon')) {
                    showModal('لطفاً یک لینک معتبر از سایت آمازون وارد کنید.');
                    return;
                }

                if (!priceString) {
                     showModal(`لطفاً قیمت محصول به ${settings.currency_symbol} را وارد کنید.`);
                    return;
                }
                const cleanedPriceString = priceString.replace(/,/g, '');
                const priceAED = parseFloat(cleanedPriceString);
                if (isNaN(priceAED) || priceAED <= 0) {
                    showModal(`لطفاً قیمت محصول به ${settings.currency_symbol} را به درستی وارد کنید.`);
                    return;
                }

                formSection.hide();
                resultSection.hide();
                loadingSection.show();
                
                // Mock API call
                setTimeout(() => {
                    const productDetails = { title: 'محصول شما از آمازون', priceAED: priceAED };
                    
                    const exchangeRate = parseFloat(settings.exchange_rate);
                    const profitPercentage = parseFloat(settings.profit_percentage);

                    if (isNaN(exchangeRate) || isNaN(profitPercentage)) {
                        showModal('خطا در تنظیمات افزونه. لطفاً با مدیر سایت تماس بگیرید.');
                        loadingSection.hide();
                        formSection.show();
                        return;
                    }

                    const basePriceToman = priceAED * exchangeRate;
                    const finalPrice = basePriceToman * (1 + (profitPercentage / 100));

                    displayResults(productDetails, finalPrice);
                }, 500);
            });

            airShippingOption.on('click', function() {
                selectedShippingMethod = 'air';
                updateShippingSelectionVisuals();
            });

            seaShippingOption.on('click', function() {
                selectedShippingMethod = 'sea';
                updateShippingSelectionVisuals();
            });

            whatsappBtn.on('click', function() {
                if (!currentProductData.title) return;

                const selectedMethodName = selectedShippingMethod === 'air' ? settings.shipping_air_text : settings.shipping_sea_text;
                
                const message = `سلام،
من مایلم این محصول را سفارش دهم:

نام محصول: ${currentProductData.title}
قیمت به ${settings.currency_symbol}: ${currentProductData.priceAED.toLocaleString('en-US', { minimumFractionDigits: 2, maximumFractionDigits: 2 })}

روش ارسال انتخابی: ${selectedMethodName}
قیمت نهایی با کارمزد: ${currentProductData.finalPriceToman} تومان

لینک محصول: ${amazonUrlInput.val()}

لطفاً برای ادامه راهنمایی کنید.`;

                const whatsappUrl = `https://wa.me/${settings.whatsapp_number}?text=${encodeURIComponent(message)}`;
                window.open(whatsappUrl, '_blank');
            });
        });
    });
})(jQuery);
