<?php
/**
 * Class Amazon_Price_Calculator_Frontend
 *
 * مدیریت تمام فعالیت‌های مربوط به بخش کاربری سایت.
 */
class Amazon_Price_Calculator_Frontend {

    /**
     * @var array فیلدهای تنظیمات افزونه.
     */
    private $settings_fields;
    
    /**
     * @var array گزینه‌های ذخیره شده در دیتابیس با مقادیر پیش‌فرض.
     */
    private $options;

    /**
     * سازنده کلاس.
     *
     * @param array $settings_fields آرایه‌ای از فیلدهای تنظیمات.
     */
    public function __construct($settings_fields) {
        $this->settings_fields = $settings_fields;
    }

    /**
     * افزودن فایل‌های CSS و JS به صفحات سایت.
     */
    public function enqueue_assets() {
        if ( is_admin() ) {
            return;
        }
        
        // فقط در صورتی که شورت‌کد در صفحه وجود داشته باشد، فایل‌ها را بارگذاری کن.
        global $post;
        if ( is_a( $post, 'WP_Post' ) && has_shortcode( $post->post_content, 'amazon_price_calculator' ) ) {
            wp_enqueue_style('google-fonts-vazirmatn', 'https://fonts.googleapis.com/css2?family=Vazirmatn:wght@400;500;700&display=swap', array(), null);
            wp_enqueue_style('apc-frontend-style', APC_URL . 'assets/css/apc-frontend-style.css', array('google-fonts-vazirmatn'), APC_VERSION);
            
            wp_enqueue_script('apc-frontend-script', APC_URL . 'assets/js/apc-frontend-script.js', array('jquery'), APC_VERSION, true);
            wp_localize_script('apc-frontend-script', 'apc_settings', $this->get_localized_settings());
        }
    }

    /**
     * رندر کردن خروجی شورت‌کد.
     *
     * @param array $atts آرگومان‌های شورت‌کد.
     * @return string خروجی HTML شورت‌کد.
     */
    public function render_shortcode($atts) {
        $opts = $this->get_options_with_defaults();
        $unique_id = 'apc-instance-' . uniqid();

        $main_color = esc_attr($opts['main_color']);
        $max_width = esc_attr($opts['max_width']);
        $body_font_size = esc_attr($opts['body_font_size']);
        $title_font_size = esc_attr($opts['title_font_size']);
        $subtitle_font_size = esc_attr($opts['subtitle_font_size']);
        $button_font_size = esc_attr($opts['button_font_size']);

        ob_start();
        ?>
        <style>
            #<?php echo $unique_id; ?>.apc-container-wrapper {
                all: initial !important;
                display: block !important;
                font-family: 'Vazirmatn', sans-serif !important;
                direction: rtl !important;
            }
            #<?php echo $unique_id; ?> .apc-container {
                margin: 2rem auto !important;
                max-width: <?php echo $max_width; ?>px !important;
                --apc-main-color: <?php echo $main_color; ?>;
                font-size: <?php echo $body_font_size; ?>px;
            }
            #<?php echo $unique_id; ?> .apc-card-header h1,
            #<?php echo $unique_id; ?> .apc-card-header h2 {
                font-size: <?php echo $title_font_size; ?>px;
            }
            #<?php echo $unique_id; ?> .apc-card-header p {
                font-size: <?php echo $subtitle_font_size; ?>px;
            }
            #<?php echo $unique_id; ?> .apc-btn {
                font-size: <?php echo $button_font_size; ?>px;
            }
        </style>

        <div id="<?php echo $unique_id; ?>" class="apc-container-wrapper">
            <div class="apc-container">
                <div class="apc-form-section apc-card">
                    <div class="apc-card-header">
                        <h1><?php echo esc_html($opts['form_title']); ?></h1>
                        <p>
                            <?php echo esc_html($opts['form_subtitle']); ?>
                            <a href="<?php echo esc_url($opts['source_site_url']); ?>" target="_blank">(<?php echo esc_html($opts['source_site_text']); ?>)</a>
                        </p>
                    </div>
                    <div class="apc-card-body">
                        <input type="url" class="apc-amazon-url-input" placeholder="<?php echo esc_attr($opts['url_placeholder']); ?>">
                        <input type="text" class="apc-aed-price-input" placeholder="<?php echo esc_attr($opts['price_placeholder']); ?>" inputmode="decimal">
                        <button class="apc-btn apc-calculate-btn">
                            <?php echo esc_html($opts['button_text']); ?>
                        </button>
                    </div>
                </div>

                <div class="apc-loading-section apc-card" style="display: none;">
                    <div class="apc-loader"></div>
                    <p>در حال محاسبه قیمت‌ها...</p>
                </div>

                <div class="apc-result-section apc-card" style="display: none;">
                    <div class="apc-card-header">
                        <h2><?php echo esc_html($opts['results_title']); ?></h2>
                    </div>
                    <div class="apc-card-body">
                        <div class="apc-result-row">
                            <span>قیمت کالا به <?php echo esc_html($opts['currency_symbol']); ?>:</span>
                            <span class="apc-product-price-aed"></span>
                        </div>
                        <div class="apc-result-row apc-exchange-rate">
                            <p class="apc-exchange-rate-display"></p>
                        </div>
                        <hr>
                        <p class="apc-shipping-title">نوع ارسال را انتخاب کنید:</p>
                        <div class="apc-shipping-option apc-air-shipping-option apc-selected">
                            <div class="apc-shipping-info">
                                <div><?php echo esc_html($opts['shipping_air_text']); ?></div>
                                <div class="apc-air-price"></div>
                            </div>
                        </div>
                        <div class="apc-shipping-option apc-sea-shipping-option">
                           <div class="apc-shipping-info">
                                <div><?php echo esc_html($opts['shipping_sea_text']); ?></div>
                                <div class="apc-sea-price"></div>
                            </div>
                        </div>
                        <hr>
                        <button class="apc-btn apc-whatsapp-btn">
                            <?php echo esc_html($opts['whatsapp_button_text']); ?>
                        </button>
                        <p class="apc-footer-text"><?php echo esc_html($opts['footer_text']); ?></p>
                    </div>
                </div>
            </div>
        </div>
        <?php
        return ob_get_clean();
    }

    /**
     * آماده‌سازی تنظیمات برای ارسال به جاوااسکریپت.
     * @return array
     */
    private function get_localized_settings() {
        return $this->get_options_with_defaults();
    }
    
    /**
     * دریافت تنظیمات ذخیره شده و ادغام آن‌ها با مقادیر پیش‌فرض.
     * @return array
     */
    private function get_options_with_defaults() {
        if ( ! isset($this->options) ) {
            $db_options = get_option('apc_settings');
            $defaults = array_column($this->settings_fields, 'default', 'id');
            $this->options = wp_parse_args($db_options, $defaults);
        }
        return $this->options;
    }
}
